// src/utils/request.js

import axios from 'axios';
import {
	Message,
	Loading
} from 'element-ui';
import store from '../store' // 直接导入Store
import router from '../router'; // 你的路由实例

let loadingInstance = null;

// 创建 axios 实例
const service = axios.create({
	baseURL: process.env.VUE_APP_BASE_API, // 环境变量中配置的API地址
	timeout: 150000 // 请求超时时间
});

// 请求拦截器
service.interceptors.request.use(
	config => {
		// 显示 loading（可根据需要配置）
		if (config.loading !== false) {
			loadingInstance = Loading.service({
				lock: true,
				text: '加载中...',
				background: 'rgba(0, 0, 0, 0.7)'
			});
		}

		// 添加认证 token（如果有）
		const token = localStorage.getItem('token');
		if (token) {
			config.headers.Authorization = `Bearer ${token}`;
		}

		// 处理特殊 Content-Type
		if (config.formData) {
			config.headers['Content-Type'] = 'multipart/form-data';
		} else {
			config.headers['Content-Type'] = 'application/x-www-form-urlencoded';
		}
		if (config.responseType) {
			config.headers['responseType'] = config.responseType;
		}
		console.log(token)
		if (token) {
			if (!config.data) {
				config.data = {};
			}
			config.data.token = token
		}
		return config;
	},
	error => {
		// 关闭 loading
		loadingInstance && loadingInstance.close();
		return Promise.reject(error);
	}
);

// 响应拦截器
service.interceptors.response.use(
	response => {
		// 关闭 loading
		loadingInstance && loadingInstance.close();
		if (response.request.responseType === 'blob') {
			return response;
		}
		const res = response.data;
		// 自定义状态码处理（根据后端接口约定修改）
		if (res.code !== 1) {
			Message({
				message: res.msg || '请求出错，请稍后重试',
				type: 'error',
				duration: 5 * 1000
			});

			// 特殊状态码处理
			if (res.code === 401) {
				// 登录过期处理
				store.state.userinfo = null;
				localStorage.removeItem('userinfo');
				store.state.token = null;
				localStorage.removeItem('token');
				router.push('login');
			}

			// return res.msg;
			return Promise.reject(new Error(res.msg || 'Error'));
		}

		return res;
	},
	error => {
		// 关闭 loading
		loadingInstance && loadingInstance.close();

		// HTTP 状态码处理
		if (error.response) {
			switch (error.response.status) {
				case 400:
					error.message = '请求错误';
					break;
				case 401:
					error.message = '登录已过期，请重新登录';
					store.state.userinfo = null;
					localStorage.removeItem('userinfo');
					store.state.token = null;
					localStorage.removeItem('token');
					router.push('/login');
					break;
				case 403:
					error.message = '拒绝访问';
					break;
				case 404:
					error.message = '请求资源不存在';
					break;
				case 500:
					error.message = '服务器内部错误';
					break;
				default:
					error.message = `连接错误 ${error.response.status}`;
			}
		} else {
			error.message = '无法连接服务器，请检查网络';
		}

		Message({
			message: error.message,
			type: 'error',
			duration: 5 * 1000
		});

		return Promise.reject(error);
	}
);

/**
 * 封装请求方法
 * @param {Object} config 请求配置
 * @returns {Promise}
 */
export function request(config) {
	return service(config);
}

/**
 * 封装 GET 请求
 * @param {String} url 
 * @param {Object} params 
 * @param {Object} options 
 * @returns {Promise}
 */
export function get(url, params = {}, options = {}) {
	return service({
		url,
		method: 'get',
		params,
		...options
	});
}

/**
 * 封装 POST 请求
 * @param {String} url 
 * @param {Object} data 
 * @param {Object} options 
 * @returns {Promise}
 */
export function post(url, data = {}, options = {}) {
	return service({
		url,
		method: 'post',
		data,
		...options
	});
}

// 其他方法（PUT、DELETE等）类似...